close all
clear
clc

rand('seed',0);
randn('seed',0);

disp('Test: Sequential estimator')
disp('--------------------------')
%% Input parameters
D = 2; %dimensions

aux_x = 10*rand(6, 1) + 1; % aux nodes and self node
aux_y = 10*rand(6, 1) + 1; 

x = [1 11  1  11 aux_x' ]'; 
y = [1  1  11 11 aux_y' ]';

N   = length(x);        % total # nodes
N_u = length(aux_x);    % # unknown-position nodes
N_a = N - N_u;          % # anchors (with prior)
N_ab = 4;               % # anchors used in each auxiliary batch

% SCHEDULE for the general case of N nodes, 
[ T, B ] = func_gen_seq_batch( N, N_u, N_ab, [N, (N-1):-1:(N-(N_u-1))] );

print_seq_batch(T, N_ab);

%index of self-localizing receiver-only node (must be the last one)
N_select = max(T)+1; 

X_mean = [x y]';
mu_theta = reshape(X_mean, 2*N, 1); % mean positions of all nodes

%Signal parameters
sigma2 = (10E-10)^2; % variance of the time-of-flight measurement, in seconds^2
c      = 3E8; % speed of light 

K = N*(N-1)/2; % number of node-to-node distances

% # of nodes on which CRB is calculated 
N_crb = N_u; 


%% Prior parameters

%Positions prior (#N)
sigma2_theta = [3e-1*ones(1,N_a) 1e2*ones(1,N_u)].^2; %[m^2]
mu_theta     = [mu_theta(1:N_a*D); 100*ones(D*N_u,1)];

%Delay prior (#N-1)
sigma2_delta = (1e-10*ones(1,N-1)).^2; %[s^2]
mu_delta     = 1e-6*ones(N-1,1);

%Construct augmented prior
mu    = [mu_theta; mu_delta];

P_theta_diag = repmat( sigma2_theta, D, 1 );
P_theta_diag = P_theta_diag(:);
P_theta      = diag(P_theta_diag);

P_delta = diag(sigma2_delta);

P     = blkdiag(P_theta, P_delta);
P_inv = inv(P);

%Compute observation and covariance matrices @ N_select
[H, ~] = func_create_matrices_self( T, N_select, c );
fprintf('Rank of observation matrix: %d\n', rank(full(H)));

% covariance matrix
M=size(H,1); 
Q = diag(ones(1,M))+diag(1/3*ones(1,M-1),1)+diag(1/3*ones(1,M-1),-1);
Q_inv = inv(Q);


%% INIT and MAP

%Allocate
N_MC            = 1e1; % number of Monte Carlo iterations
X_MAP_vec       = zeros(size(X_mean, 1), size(X_mean, 2), N_MC); 
X_true          = zeros(size(X_mean, 1), size(X_mean, 2), N_MC);
sigma2_hat_vec  = zeros(N_MC, 1);
J_D_bar         = zeros(N*D+N, N*D+N, N_MC);
tr_SE_MAP       = zeros(N_MC,1);
tr_SE_MAP_delta = zeros(N_MC,1);
count_conv      = zeros(N_MC,1);

tic
for MC_idx = 1 : N_MC,
    
    %Random anchor positions (model uncertainty)
    theta_true              = mvnrnd( mu_theta, P_theta )';
    theta_true(D*N_a+1:end) = reshape(X_mean(:,N_a+1:end), D*N_u, 1);    %deterministic parameters
    X_true(:,:,MC_idx)      = reshape(theta_true, D, N); %reshape
    
    %Random anchor delays (model uncertainty)
    delta_true = mvnrnd( mu_delta, P_delta )';
    
    %Compute J_D_bar
    J_D_bar(:,:,MC_idx) = func_compute_J_D_bar_fast( X_true(:,:,MC_idx), sigma2, N, D, c, H, Q_inv );
    
    %Generate timing measurements
    rho_true   = pdist(X_true(:,:,MC_idx)','euclidean')'; % true node-to-node distances
    y_measured = mvnrnd(1/c*H*[rho_true; delta_true], sigma2*Q)'; % correlated noise
    
    %Initialize with INIT estimate
    X_centroid = [X_mean(:,1:N_a) [6*ones(1,N_u-1); 4*ones(1,N_u-1)] mean(X_mean(:,1:N_a), 2)]; % centroid initialization
    theta_init = X_centroid(:);
        
    %MAP estimator
    tol_theta     = 1e-2 * N_u / 1000;
    vartheta_init = [theta_init; mu_delta];

    [vartheta_hat, sigma2_hat, count_conv(MC_idx), mean_inner(MC_idx)] = ...
        func_iterativeMAP_seq_batch_alt_fast( vartheta_init, y_measured, H, N, D, P, tol_theta, c, B, Q_inv, X_true(:,:,MC_idx), N_a );


    X_MAP_vec(:,:,MC_idx)  = reshape(vartheta_hat(1:D*N), 2, N);
    X_MAP_error = squeeze(X_MAP_vec(:,N_a+1:N,MC_idx) - X_true(:,N_a+1:N,MC_idx));
    tr_SE_MAP(MC_idx) = trace(X_MAP_error*X_MAP_error'); % trace of squared error
    
    %trace of squared error of delta
    delta_MAP_error = vartheta_hat(D*N+1:D*N+(N-1)) - delta_true;
    tr_SE_MAP_delta(MC_idx) = trace(delta_MAP_error*delta_MAP_error');
    
    sigma2_hat_vec(MC_idx) = sigma2_hat;
    
    if mod(MC_idx,10)==0,
        fprintf('MC_idx = %d, elapsed time = %3.2f s\n', MC_idx, toc); %status display
    end
end
toc

%% Compute CRB

%Compute J_D (averaged over random variables)
J_D = mean(J_D_bar, 3);

%Compute J_P (hybrid)
J_P = zeros(N*D+N, N*D+N);
J_P(1:D*N_a, 1:D*N_a) = inv(P_theta(1:D*N_a, 1:D*N_a));
J_P(D*N+1:D*N+N-1, D*N+1:D*N+N-1) = inv(P_delta);

%Compute J_inv
J_inv = ( J_D + J_P )\eye(D*N+N);


%% Plot results
figure(1);
scatter(x(N_select), y(N_select), 100, 'r*'); % true positions of node N_select
hold on;
scatter(x(1:N_a), y(1:N_a), 100, 'kx'); % anchors
scatter(x(N_a+1 : N-1), y(N_a+1 : N-1), 100, 'g^'); % true position of aux
scatter(X_centroid(1,end), X_centroid(2,end), 100, 'ks'); % centroid
axis equal;

xlabel('x [m]');
ylabel('y [m]');  
box on;
grid on;

%  
% %Compute MSE matrix from the CRB, for each node r
alpha_level = 0.99; %'confidence level'
eta         = chi2inv(alpha_level,D); %corresponding to Gaussian in R^2

for r = 1 : N,
    
    MSE_single_node = J_inv(2*(r-1)+1:2*(r-1)+2, 2*(r-1)+1:2*(r-1)+2);
        
     z = func_error_ellipse( eta*MSE_single_node ); 
     b_pts(:,1) = x(r) + z(:,1); 
     b_pts(:,2) = y(r) + z(:,2); 
     plot( b_pts(:,1), b_pts(:,2), 'k-', 'LineWidth', 1.5 )
end

%scatter plot MAP
X_MAP_vec_x = X_MAP_vec(1,:,:);
X_MAP_vec_x = X_MAP_vec_x(:);
X_MAP_vec_y = X_MAP_vec(2,:,:);
X_MAP_vec_y = X_MAP_vec_y(:);
scatter(X_MAP_vec_x, X_MAP_vec_y,'b.');

%scatter plot true
X_true_x = X_true(1,:,:);
X_true_x = X_true_x(:);
X_true_y = X_true(2,:,:);
X_true_y = X_true_y(:);
scatter(X_true_x, X_true_y,'c.');


% plot error ellipses for MAP 
for node_idx = 1 : N
    X_MAP_error = squeeze(X_MAP_vec(:,node_idx, :) - X_true(:,node_idx, :));
    MSE_MAP(:,:,node_idx) = X_MAP_error*X_MAP_error'/N_MC;
    z = func_error_ellipse( eta*MSE_MAP(:,:,node_idx) ); 
    b_pts(:,1) = x(node_idx) + z(:,1);
    b_pts(:,2) = y(node_idx) + z(:,2);
    plot( b_pts(:,1), b_pts(:,2), 'r--', 'LineWidth', 1.2 )
end


figure; 
hist(count_conv(:), [1:10]);
xlabel('Number of iterations until convergence');
ylabel('Count');

figure; 
cdfplot(count_conv(:));
xlabel('Number of iterations until convergence');
ylabel('CDF');

mean_num_iter_convergence = mean(count_conv(:))

CRB_theta = (1/N_crb)*sqrt(trace(J_inv(D*N_a+1:D*N_a+D*N_u, D*N_a+1:D*N_a+D*N_u)))
RMSE_MAP_theta = (1/N_crb)*sqrt(mean(tr_SE_MAP,1)) 

CRB_delta = (1/N_crb)*sqrt(trace(J_inv(D*N+1:D*N+(N-1), D*N+1:D*N+(N-1))))
RMSE_MAP_delta = (1/N_crb)*sqrt(mean(tr_SE_MAP_delta,1)) 

average_sigma_hat = mean(sqrt(sigma2_hat))
true_sigma = sqrt(sigma2)

average_inner = mean(mean_inner)

