function [ T, B ] = func_gen_seq_batch( N, N_u, N_ab, T_u )
% generate sequence divided in batches for localizing auxiliary nodes
%
% Alessio De Angelis
%
% Input: 
%   N: total number of nodes
%   N_u: number of unknown-position nodes (self localizing + auxiliary)
%   N_ab: number of anchors used in each auxiliary batch
%   T_u: high-level sequence of unknown-position nodes (must begin with the Nth node)
% 
% Output: 
%   T: sequence of transmissions
%   B: number of measurements in each batch
%


N_a     = N - N_u;      % # anchors
N_batch = N_ab * 2;     % # transmissions in each batch
B       = N_batch - 1;  % # measurements in each batch


T = [];

% Sanity check
if (N_a < 3) || (N_a < N_ab)
    disp('ERROR: Invalid input arguments for sequence generation');
    return
end


% First batch (anchors-only)
anchor_vec = [1 : N_a]';
for i = 1 : N_batch - 1, % count edge only once
    T = [T anchor_vec(1)];
    anchor_vec = circshift(anchor_vec,-1);
end

if T(end) == 1, % avoid ending batch with a 1 (repetition)
    T(end) = 2;
end


% Batches for auxiliary nodes
for i = 2 : length(T_u), % discard first element of T_u
    
    % interleave aux with anchors ("1 AUX 2 AUX ... ")
    for j = 1 : N_ab,
        T = [T j T_u(i)]; 
    end;
    
    if i ~= length(T_u),
        T = [T T(1:N_batch-2)]; % filler, repeat first batch
    end
    
    if T(end) == 1, % avoid ending batch with a 1 (repetition)
        T(end) = 2;
    end
end
    

return 
end