function [DensMod x NMSE] = density_model(u,y,T,delta,plot_flag)
%
%
%   Density estimation of the static transfer function 
%   of a nonlinear system with input and output complex-valued
%   entries. 
%   Please see:
%       Z. Khan, E. Zenteno, M. Isaksson, and P. Hndel, Density 
%       estimation models for strong nonlinearities in RF power amplifiers,
%       in Asia Pacific Microw. Conf., Sendai, APMC., Dec 2014, pp. .
%
%   Usage:
%     [F X NMSE] = density_model(u,y,T,delta,plot_flag)
%
%   Inputs: 
%      u: (vector) complex-valued input data for the identification
%      y: (vector) complex-valued output
%      T: (scalar integer) number of estimation points for the real and 
%                imaginary parts (suggested values in the interval [20 70])
%                total number of points is T^2. (Default = 30)
%      delta: (scalar) kernel aperture (window size). default = 1.5% of the
%                input signal span.  
%      plot_flag: (scalar) if plot_flag==1, show the transfer function,
%                AM and PM deviations from the linear (expected) behavior.
%                default = 0 (not show transfer functions). 
%
%   Outputs:
%      F: (matrix)contains the values of the nonlinear transfer function.
%                 returns a NAN where any sample is found to estimate
%                 the transfer function.
%      X: (vector) contains the grid of estimation points in each axis
%                real and imaginary.
%      NMSE : normalized MSE in dB passing the input data u through F.
%
%
%   Examples:
%  >>   u = rand(1e4,1)-0.5+1i*rand(1e4,1)-0.5i;
%       y = sin(3*u);                   % static function
%       [F X NMSE] = density_model(u,y,40,0.01,1);
%
%
%  >>   [F X NMSE] = density_model(u,y,30);  
%       

%
%    Efrain Zenteno  /   Zain Ahmed Khan
%    University of Gvle, Sweden
%    The Royal Institute of Technology KTH, Sweden
%    2014/09/04
%

%% input default settings
if nargin == 2
    T = 60;
    delta = 0.015*max(abs(u));
    plot_flag = 0;
elseif nargin == 3
    delta = 0.01*max(abs(u));
    plot_flag = 0;
elseif nargin == 4
    plot_flag = 0;
end


%%
xmax = max(abs(u));
tmp = 2*xmax/T;    % bandwidth of th kernel
x = linspace(0,xmax,floor(T/2));
x = [-x(end:-1:2) x];
%x = -xmax:tmp:xmax;   % range for the ident.

P2 = zeros(length(x),length(x));
 
 for mm=1:length(x)
     for ll=1:length(x)          % kernels
         tmp = [x(mm) x(ll)];  
         tmp = tmp(ones(length(u),1),:);
         tmp = tmp - [real(u) imag(u)];
         %Psi = KK2(tmp./delta);  % alternative
         Psi = sqrt(sum(((tmp./delta).^2),2));  Psi = (1-Psi).*(Psi<1);        % conical
         den = sum(Psi);
         if den ~= 0 
            P2(ll,mm) = sum(Psi.*y/den);% ...
         end
     end
 end
 
     
P2(abs(P2)==0)=nan;     % NAN to unestimated points
P2(abs(x)<1e-3,abs(x)<1e-3)=0;  % the phase at 0 amplitude is not reliable!

DensMod = P2;

% computing amplitude and phase distortion
G = u\y;
fprintf('Linear Gain calculated from data : %2.2f \n',G);
[xx yy] = meshgrid(x);
amp_dit = abs(G).*abs(xx+1i*yy)-abs(P2);
phase_dit = angle(P2)-angle(xx+1i*yy);
phase_dit(abs(phase_dit)>pi) = phase_dit(abs(phase_dit)>pi) - sign(phase_dit(abs(phase_dit)>pi))*2*pi;

% using the density model to estimate the output
y_mod_abs = interp2(x,x,abs(P2),real(u),imag(u));
y_mod_pha = interp2(x,x,phase_dit,real(u),imag(u));
    
y_mod = y_mod_abs.*exp(1i*(angle(u)+y_mod_pha));
y_mod(isnan(y_mod)==1) = y(isnan(y_mod)==1);

NMSE = db(norm(y_mod - y)/norm(y));   % normalized MSE



%% plotting
if plot_flag == 1
    figure, 
    subplot(221),surfc(x,x,abs(P2)), 
    xlabel('real'), ylabel('imag'), zlabel('Magnitude')
    
    subplot(223), surf(xx,yy,amp_dit)  
    xlabel('real'), ylabel('imag'), zlabel('Amplitude distortion')

    subplot(222), surf(x,x,180/pi*angle(P2))
    xlabel('real'), ylabel('imag'), zlabel('Phase (degrees)')

    subplot(224), surf(x,x,180/pi*(phase_dit))
    xlabel('real'), ylabel('imag'), zlabel('Phase distortion (degreees)')
end

    
    