% Program for reproducing results in the paper 'A Universal Algorithm for 
% Compressive Sampling Reconstruction and RIP-based Analysis'
% This program, by deafult, runs for Gaussian sparse signal, at 20 dB SMNR
% Comparing the performance between OMP, SP, BPDN, UACS-1 and UACS-2

% Important Note : Please install cvx package before running this program.

function [] = Algorithm_comparison_script(SMNR, str_type, range)

% signal lengths
N = 500;
% number of spikes in the signal
K = 20;
% % number of observations to make
% M = 100;

% How many times the iteration of OMP should proceed
K_Max=20;

% No of 'Sensing Matrix' Simulations
NoOfSensingMatrix=100;

% No of data points
NoOfData=100;

% Non-Gaussian mean
% Mean_a=0.2234;    % This is for the source like (-a)-0-(a) (Using a GMM of two Gaussians)
% Mean_a=0;           % This is a Gaussian source
% Mean_a=100000;      % A Large number; Then it is for only binary source 

if strcmp(str_type,'Gaussian')
    Mean_a = 0;
elseif strcmp(str_type,'Binary')
    Mean_a = 100000;
end

% Signal Power
if (Mean_a ~= 100000)
    Signal_P=1;    % Variance of approximately sparse signal
    p=K/N;
    zeta=0.00000000000001;         % in percentage; 'K' samples have (1-zeta) power, and 'N-K' samples have zeta power
    sigma_2= ((1-(zeta/100))*Signal_P)/(p*N) - Mean_a^2;
    gamma_2= ((zeta/100)*Signal_P)/((1-p)*N);
end



% Signal to measurement noise ratio (SMNR) in dB
% SMNR=20; % in dB 

% No of measurements using Alpha
Alpha_min=range(1);
Alpha_max=range(end);
Alpha_increment=(Alpha_max-Alpha_min)/(length(range)-1);

% --------------- First Loop -------------------
Alpha=Alpha_min:Alpha_increment:Alpha_max; M_array=zeros(1,length(Alpha));
Alpha_length=length(Alpha); clear Alpha;

TotalData=NoOfSensingMatrix*NoOfData;


% MSE error
Recon_error_OMP=zeros(TotalData,Alpha_length);
Recon_error_SP=zeros(TotalData,Alpha_length);
Recon_error_BPDN=zeros(TotalData,Alpha_length);
Recon_error_uacs1=zeros(TotalData,Alpha_length);
Recon_error_uacs2=zeros(TotalData,Alpha_length);

% For keeping the cardinality of  I_true intersect I_estimate
Card_OMP_Mat=zeros(TotalData,Alpha_length);
Card_SP_Mat=zeros(TotalData,Alpha_length);
Card_BPDN_Mat=zeros(TotalData,Alpha_length);
Card_uacs1_Mat=zeros(TotalData,Alpha_length);
Card_uacs2_Mat=zeros(TotalData,Alpha_length);

kkk=1;    
    
%% -------- First Loop ------------

for Alpha=Alpha_min:Alpha_increment:Alpha_max
    
    data_index=1;
    
%     Alpha=Alpha_min;
    Alpha
    
    
    % Fixing M (no of measurements)
    M=round(Alpha*N);
    M_array(kkk)=M;
    
    
%% ---------- Second Loop --------------
    
    for Sensing_instance=1:NoOfSensingMatrix
    
        
        % Sparse data creation
        if (Mean_a==100000),
            [X U]=SparseDataCreate_ZeroOne(NoOfData,N,K);
            Signal_P=K;
        else
            [X U]=SparseDataCreate_G_to_NonG(NoOfData,N,K,sigma_2,gamma_2,Mean_a);
        end
            
    
        
        % measurement matrix
        disp('Creating measurement matrix...');
        A = randn(M,N);
        %     A = orth(A')';
        for n=1:N
            A(:,n)=A(:,n)/norm(A(:,n));
        end
        %     A=A/sqrt(M);
        disp('Done.');
        
        
        % CS Measurements
        Y= (A * X')';
        
        
        % Measurement noise simulation 
        sigma_2_measurement_noise = Signal_P / (M*(10^(SMNR/10)));
        Mu_Noise=zeros(M,1); Cov_Noise=diag(sigma_2_measurement_noise*ones(M,1));
        NoiseMatrix=mvnrnd(Mu_Noise,Cov_Noise,NoOfData);
        
        
        % Noisy CS measurements
        Z=Y+NoiseMatrix;    
        
        
        % epsilon for BPDN
        lambda=2;
        epsilon=sqrt(sigma_2_measurement_noise * (M+(lambda*sqrt(2*M))) );
        
        
        
%% -------------- Third loop ---------------------
        for i=1:NoOfData
            
            % observations
            y = Z(i,:)';
            
            
                                
            % -------- Computationally Efficient OMP algo ------
            [xp_OMP, I_OMP, y_r_OMP]=OMP_func_CompEff(y,A,N,K_Max);
            
            
            % ------- Subspace Pursuit ----------------
            [xp_SP, I_SP, y_r_SP]=Subspace_pursuit_func(y,A,K,N);
            
             % ------- UACS-1 : OMP+SP ----------------
            [xp_uacs1, I_uacs1, y_r_uacs1]=iter_omp_sp(y,A,K);
   
            % ------- UACS-2 : OMP+SP+BPDN  ----------------

            [xp_uacs2, I_uacs2, y_r_uacs2]=iter_l1_omp_sp(y,A,K,epsilon);


            % -------- BPDN (L1) ---------

            [xp_BPDN, I_BPDN, y_r_BPDN]=simple_BP(y,A,K,epsilon);
            
            
            
            
            % ------- Support Set Estimation Dependent Oracle Assisted Performances (BPDN) ------
            
            % For BPDN
            I_est_BPDN=support_set_eval(xp_BPDN,K);
            A_I_BPDN=A(:,I_est_BPDN);
            xp_high_amplitude=pinv(A_I_BPDN) * y;
            xp_Oracle_BPDN=zeros(N,1);
            xp_Oracle_BPDN(I_est_BPDN)=xp_high_amplitude;
            
            % ---------------- PERFORMANCE CALCULATION --------------------
            
            % reconstruction errors
            error_OMP=(norm(X(i,:)-xp_OMP')^2);
            error_SP=(norm(X(i,:)-xp_SP')^2);
            error_uacs1=(norm(X(i,:)-xp_uacs1')^2);
            error_uacs2=(norm(X(i,:)-xp_uacs2')^2);
            error_BPDN=(norm(X(i,:)-xp_Oracle_BPDN')^2);            
            
            Recon_error_OMP(data_index,kkk)=error_OMP;
            Recon_error_SP(data_index,kkk)=error_SP;
            Recon_error_uacs1(data_index,kkk)=error_uacs1;
            Recon_error_uacs2(data_index,kkk)=error_uacs2;
            Recon_error_BPDN(data_index,kkk)=error_BPDN;
            
            
            % Cardinality of the intersection set of true support and estimated support
            I_true=support_set_eval(X(i,:)',K);
            
            I_est_OMP=support_set_eval(xp_OMP,K);
            I_est_SP=support_set_eval(xp_SP,K);
            I_est_uacs1=support_set_eval(xp_uacs1,K);
            I_est_uacs2=support_set_eval(xp_uacs2,K);
            I_est_BPDN=I_est_BPDN;
            
            Card_OMP_Mat(data_index,kkk)=length(intersect(I_true,I_est_OMP)); 
            Card_SP_Mat(data_index,kkk)=length(intersect(I_true,I_est_SP));
            Card_uacs1_Mat(data_index,kkk)=length(intersect(I_true,I_est_uacs1));
            Card_uacs2_Mat(data_index,kkk)=length(intersect(I_true,I_est_uacs2));
            Card_BPDN_Mat(data_index,kkk)=length(intersect(I_true,I_est_BPDN));
                  
            
            data_index=data_index+1;
            
        end
%% end of the Third 'for' loop        
        
    end
%% end of the Second 'for' loop

kkk=kkk+1;



%% Performance evaluation using signal to reconstruction noise error (SRNR)
% in dB
avg_Recon_error_OMP=mean(Recon_error_OMP);
avg_Recon_error_SP=mean(Recon_error_SP);
avg_Recon_error_uacs1=mean(Recon_error_uacs1);
avg_Recon_error_uacs2=mean(Recon_error_uacs2);
avg_Recon_error_BPDN=mean(Recon_error_BPDN);

SRNR_OMP=10*log10(Signal_P ./ avg_Recon_error_OMP);
SRNR_SP=10*log10(Signal_P ./ avg_Recon_error_SP);
SRNR_uacs1=10*log10(Signal_P ./ avg_Recon_error_uacs1);
SRNR_uacs2=10*log10(Signal_P ./ avg_Recon_error_uacs2);
SRNR_BPDN=10*log10(Signal_P ./ avg_Recon_error_BPDN);

% Writing into a file
str1=cat(2,'SRNR_for_',str_type);
str1=cat(2,str1,'_Signal_at_SMNR_');
str2=int2str(SMNR);
str1=cat(2,str1,str2);
str3='.txt';
str1=cat(2,str1,str3);

fp=fopen(str1,'w');

 
for k=1:length(M_array)
    fprintf(fp,'%f ',M_array(k));
end
fprintf(fp,'\n');
for k=1:length(SRNR_OMP)
    fprintf(fp,'%f ',SRNR_OMP(k));
end
fprintf(fp,'\n');
for k=1:length(SRNR_SP)
    fprintf(fp,'%f ',SRNR_SP(k));
end
fprintf(fp,'\n');
for k=1:length(SRNR_uacs1)
    fprintf(fp,'%f ',SRNR_uacs1(k));
end
fprintf(fp,'\n');
for k=1:length(SRNR_uacs2)
    fprintf(fp,'%f ',SRNR_uacs2(k));
end
fprintf(fp,'\n');
for k=1:length(SRNR_BPDN)
    fprintf(fp,'%f ',SRNR_BPDN(k));
end


%% Performance evaluation using the intersection set of true support and
%% estimated support
% in ratio

avg_Card_OMP=mean(Card_OMP_Mat);
avg_Card_SP=mean(Card_SP_Mat);
avg_Card_uacs1=mean(Card_uacs1_Mat);
avg_Card_uacs2=mean(Card_uacs2_Mat);
avg_Card_BPDN=mean(Card_BPDN_Mat);

ASCE_OMP=1-avg_Card_OMP/K;
ASCE_SP=1-avg_Card_SP/K;
ASCE_uacs1=1-avg_Card_uacs1/K;
ASCE_uacs2=1-avg_Card_uacs2/K;
ASCE_BPDN=1-avg_Card_BPDN/K;

% Writing into a file
str1=cat(2,'ASCE_for_',str_type);
str1=cat(2,str1,'_Signal_at_SMNR_');
str2=int2str(SMNR);
str1=cat(2,str1,str2);
str3='.txt';
str1=cat(2,str1,str3);

fp=fopen(str1,'w');
 
for k=1:length(M_array)
    fprintf(fp,'%f ',M_array(k));
end
fprintf(fp,'\n');
for k=1:length(ASCE_OMP)
    fprintf(fp,'%f ',ASCE_OMP(k));
end
fprintf(fp,'\n');
for k=1:length(ASCE_SP)
    fprintf(fp,'%f ',ASCE_SP(k));
end
fprintf(fp,'\n');
for k=1:length(ASCE_uacs1)
    fprintf(fp,'%f ',ASCE_uacs1(k));
end
fprintf(fp,'\n');
for k=1:length(ASCE_uacs2)
    fprintf(fp,'%f ',ASCE_uacs2(k));
end
fprintf(fp,'\n');
for k=1:length(ASCE_BPDN)
    fprintf(fp,'%f ',ASCE_BPDN(k));
end

end
%% end of the first 'for' loop
fclose('all');
return